<?php
/* --------------------------------------------------------------
   CronjobTaskRepository.inc.php 2019-02-15
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2019 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

/**
 * Class CronjobTaskRepository
 */
class CronjobTaskRepository implements CronjobTaskRepositoryInterface
{
    /**
     * @var \CronjobSettings
     */
    protected $settings;
    
    protected $storage;
    
    
    /**
     * CronjobTaskRepository constructor.
     *
     * @param \CronjobSettings             $settings
     * @param \CronjobConfigurationStorage $storage
     */
    public function __construct(CronjobSettings $settings, CronjobConfigurationStorage $storage)
    {
        $this->settings = $settings;
        $this->storage  = $storage;
    }
    
    
    /**
     * Returns all cronjob tasks.
     *
     * @return \CronjobTaskCollection
     * @throws \CronjobConfigurationNotFoundException
     */
    public function getAll()
    {
        $iterator = new IteratorIterator(new DirectoryIterator($this->settings->configurationDirectory()));
        $tasks    = [];
        
        foreach ($iterator as $configFile) {
            /** @var \SplFileInfo $configFile */
            if ($this->_isCronjobConfiguration($configFile)) {
                $cronjobName = str_replace('.json', '', $configFile->getFilename());
                
                $loggerName       = $cronjobName . 'CronjobLogger';
                $dependenciesName = $cronjobName . 'CronjobDependencies';
                $taskName         = $cronjobName . 'CronjobTask';
                
                $logger       = MainFactory::create($loggerName, new ExistingDirectory($this->settings->getRoot()));
                $dependencies = MainFactory::create($dependenciesName, $this->storage);
                
                $schedule  = new StringType($this->storage->get($cronjobName, 'interval'));
                $sortOrder = new IntType($this->storage->get($cronjobName, 'sortOrder'));
                
                $active = new BoolType($this->storage->get($cronjobName, 'active'));
                if ($active->asBool()) {
                    $tasks[] = MainFactory::create($taskName, $schedule, $sortOrder, $logger, $dependencies);
                }
            }
        }
        
        $tasks = count($tasks) === 0 ? [] : $tasks;
        
        return CronjobTaskCollection::collect(...$tasks);
    }
    
    
    /**
     * Checks if given file info is a cronjob configuration file.
     *
     * @param \SplFileInfo $configFile
     *
     * @return bool
     */
    protected function _isCronjobConfiguration(SplFileInfo $configFile)
    {
        return strpos($configFile->getFilename(), '.json') !== false;
    }
}